<?php
/**
 * Plugin Name: upmlm-connect
 * Description: Free plug & play integration between WordPress forms and UpMLM CRM.
 * Support: Contact client@upmlm.com | https://upmlm.com
 * Version: 0.5.4
 * Author: UpMLM
 * Text Domain: upmlmconnect
 * Domain Path: /languages
 */

if (!defined('ABSPATH'))
    exit;

class UpMlmConnect
{

    const OPTION_KEY = 'upmlmconnect_settings';
    const LOG_FILENAME = 'upmlmconnect-errors.log';
    const STATUS_KEY = 'upmlmconnect_last_status';

    public function __construct()
    {
        add_action('plugins_loaded', [$this, 'load_textdomain']);
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('init', [$this, 'hook_forms']);

        // NEW
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_settings_link']);
        add_action('admin_notices', [$this, 'missing_config_notice']);
    }

    public function load_textdomain()
    {
        load_plugin_textdomain('upmlmconnect', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /* =====================
     * ADMIN
     * ===================== */

    public function register_admin_menu()
    {
        add_menu_page(
            __('UpMLM Connect', 'upmlmconnect'),
            __('UpMLM Connect', 'upmlmconnect'),
            'manage_options',
            'upmlmconnect',
            [$this, 'settings_page'],
            'dashicons-rest-api'
        );
    }

    public function register_settings()
    {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [$this, 'sanitize_settings']);
    }

    /**
     * NEW: enable all providers by default
     */
    public function sanitize_settings($settings)
    {
        if (empty($settings['providers'])) {
            $settings['providers'] = [
                'cf7' => 1,
                'elementor' => 1,
                'wpforms' => 1,
                'gravityforms' => 1,
                'forminator' => 1,
                'ninjaforms' => 1,
            ];
        }
        return $settings;
    }

    /**
     * NEW: Settings link in plugin list
     */
    public function add_settings_link($links)
    {
        $url = admin_url('admin.php?page=upmlmconnect');
        array_unshift($links, '<a href="' . esc_url($url) . '">' . esc_html__('Settings', 'upmlmconnect') . '</a>');
        return $links;
    }

    /**
     * NEW: Admin notice if config missing
     */
    public function missing_config_notice()
    {
        if (!current_user_can('manage_options'))
            return;

        $settings = get_option(self::OPTION_KEY);
        if (!empty($settings['api_key']) && !empty($settings['org_id']))
            return;

        $link = '<a href="' . esc_url(admin_url('admin.php?page=upmlmconnect')) . '">' .
            esc_html__('Configure UpMLM Connect', 'upmlmconnect') . '</a>';

        echo '<div class="notice notice-warning is-dismissible"><p>' .
            wp_kses_post(sprintf(
                __('UpMLM Connect is almost ready. Please configure your API Key and Organization ID. %s', 'upmlmconnect'),
                $link
            )) .
            '</p></div>';
    }

    private function link($url, $label = null)
    {
        $label = $label ?: $url;
        return '<a href="' . esc_url($url) . '" target="_blank" rel="noopener">' . esc_html($label) . '</a>';
    }


    private function build_about($form_label)
    {
        $site_title = get_bloginfo('name');
        $label = trim((string) $form_label);
        if ($label === '')
            $label = 'Form';
        return 'WordPress lead: ' . $site_title . ' - ' . $label;
    }
    private function card_open($title = '')
    {
        echo '<div class="postbox" style="max-width: 920px;">';
        if ($title !== '') {
            echo '<h2 class="hndle" style="padding: 10px 12px; margin:0;"><span>' . esc_html($title) . '</span></h2>';
        }
        echo '<div class="inside" style="padding: 12px;">';
    }

    private function card_close()
    {
        echo '</div></div>';
    }

    public function settings_page()
    {
        if (!current_user_can('manage_options'))
            return;

        $settings = get_option(self::OPTION_KEY);
        $providers = $settings['providers'] ?? [
            'cf7' => 1,
            'elementor' => 1,
            'wpforms' => 1,
            'gravityforms' => 1,
            'forminator' => 1,
            'ninjaforms' => 1,
        ];

        // Handle log actions
        if (isset($_POST['upmlm_log_action']) && check_admin_referer('upmlm_log_action')) {
            $action = sanitize_text_field($_POST['upmlm_log_action']);
            if ($action === 'clear') {
                $this->clear_log();
                echo '<div class="notice notice-success"><p>' . esc_html__('Log cleared.', 'upmlmconnect') . '</p></div>';
            }
        }

        // Handle test
        $test_result_notice = null;
        if (isset($_POST['upmlm_test']) && check_admin_referer('upmlm_test_lead')) {
            $ok = $this->send_to_upmlm([
                'name' => 'WordPress Test',
                'email' => 'test@upmlm.com'
            ], false, 'admin_test');

            $test_result_notice = $ok
                ? ['type' => 'success', 'msg' => __('Test lead sent successfully.', 'upmlmconnect')]
                : ['type' => 'error', 'msg' => __('Error contacting UpMLM API. Check your API key, Organization ID, and the error log.', 'upmlmconnect')];
        }

        echo '<div class="wrap">';
        echo '<h1 style="margin-bottom: 12px;">' . esc_html__('UpMLM Connect', 'upmlmconnect') . '</h1>';

        // Connection status
        $status = get_option(self::STATUS_KEY, []);
        if (empty($settings['api_key']) || empty($settings['org_id'])) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Not configured. Please set your API Key and Organization ID.', 'upmlmconnect') . '</p></div>';
        } elseif (!empty($status) && isset($status['status']) && $status['status'] === 'error') {
            echo '<div class="notice notice-warning"><p>' . esc_html__('Last error detected. Check the error log below.', 'upmlmconnect') . '</p></div>';
        } else {
            echo '<div class="notice notice-success"><p>' . esc_html__('Connected.', 'upmlmconnect') . '</p></div>';
        }

        // Intro help (blue left border)
        echo '<div style="background:#fff;border-left:4px solid #2271b1;padding:12px 16px;margin-bottom:16px;max-width:920px;">';
        echo '<strong>' . esc_html__('Getting started', 'upmlmconnect') . '</strong><br>';
        echo wp_kses_post(
            sprintf(
                __('See how to integrate WordPress forms with the UpMLM CRM: %s', 'upmlmconnect'),
                $this->link('https://upmlm.com/help/knowledgebase-category/worpress-integration/', __('View documentation', 'upmlmconnect'))
            )
        );
        echo '</div>';

        // Optional test notice
        if ($test_result_notice) {
            echo '<div class="notice notice-' . esc_attr($test_result_notice['type']) . '"><p>' . esc_html($test_result_notice['msg']) . '</p></div>';
        }

        // Settings form
        echo '<form method="post" action="options.php">';
        settings_fields(self::OPTION_KEY);

        $this->card_open(__('Settings', 'upmlmconnect'));

        // Description line with UpMLM link
        echo '<p>' . wp_kses_post(
            sprintf(
                __('Free plug & play integration between WordPress forms and %s CRM.', 'upmlmconnect'),
                $this->link('https://upmlm.com', 'UpMLM')
            )
        ) . '</p>';

        echo '<table class="form-table" role="presentation">';

        // Enable
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Enable integration', 'upmlmconnect') . '</th>';
        echo '<td><label><input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[enabled]" value="1" ' . checked(1, $settings['enabled'] ?? 0, false) . ' /> ' . esc_html__('Enabled', 'upmlmconnect') . '</label></td>';
        echo '</tr>';

        // API key + help
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('API Key', 'upmlmconnect') . '</th>';
        echo '<td>';
        echo '<input type="text" name="' . esc_attr(self::OPTION_KEY) . '[api_key]" value="' . esc_attr($settings['api_key'] ?? '') . '" size="55" required />';
        echo '<p class="description">' . wp_kses_post(
            sprintf(
                __('Generate your API key here: %s', 'upmlmconnect'),
                $this->link('https://app.upmlm.com/user/api-tokens', 'https://app.upmlm.com/user/api-tokens')
            )
        ) . '</p>';
        echo '</td>';
        echo '</tr>';

        // Org ID + help
        echo '<tr>';
        echo '<th scope="row">' . esc_html__('Organization ID', 'upmlmconnect') . '</th>';
        echo '<td>';
        echo '<input type="number" name="' . esc_attr(self::OPTION_KEY) . '[org_id]" value="' . esc_attr($settings['org_id'] ?? '') . '" required />';
        echo '<p class="description">';
        echo esc_html__('The Organization ID is the last number in this URL:', 'upmlmconnect') . '<br>';
        echo '<code>https://app.upmlm.com/organizations/(this number here)</code>';
        echo '</p>';
        echo '</td>';
        echo '</tr>';

        echo '</table>';

        // Providers
        echo '<hr style="margin: 16px 0;">';
        echo '<h2 style="margin: 0 0 8px 0;">' . esc_html__('Active form providers', 'upmlmconnect') . '</h2>';
        echo '<p class="description" style="max-width: 860px;">' . esc_html__('Select which form plugins should send leads to UpMLM. Name and Email are required; Phone is optional.', 'upmlmconnect') . '</p>';

        echo '<div style="background:#fff;border:1px solid #dcdcde;border-radius:10px;padding:14px;max-width:860px;">';

        // CF7 (essential help)
        echo '<div style="padding:10px 0;border-bottom:1px solid #f0f0f1;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][cf7]" value="1" ' . checked(1, $providers['cf7'] ?? 0, false) . ' /> ';
        echo esc_html__('Contact Form 7 forms', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post(
            sprintf(
                __('Essential: use field names %s, %s and %s (phone optional). %s', 'upmlmconnect'),
                '<code>your-name</code>',
                '<code>your-email</code>',
                '<code>your-phone</code>',
                $this->link('https://upmlm.com/help/knowledgebase/wordpress-contact-form-7/', __('Instructions', 'upmlmconnect'))
            )
        ) . '</div>';
        echo '</div>';

        // Elementor (essential help)
        echo '<div style="padding:10px 0;border-bottom:1px solid #f0f0f1;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][elementor]" value="1" ' . checked(1, $providers['elementor'] ?? 0, false) . ' /> ';
        echo esc_html__('Elementor forms', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post(
            sprintf(
                __('Essential: set Field IDs to %s, %s and %s (phone optional). %s', 'upmlmconnect'),
                '<code>name</code>',
                '<code>email</code>',
                '<code>phone</code>',
                $this->link('https://upmlm.com/help/knowledgebase/wordpress-elementor-form/', __('Instructions', 'upmlmconnect'))
            )
        ) . '</div>';
        echo '</div>';

        // WPForms (link only)
        echo '<div style="padding:10px 0;border-bottom:1px solid #f0f0f1;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][wpforms]" value="1" ' . checked(1, $providers['wpforms'] ?? 0, false) . ' /> ';
        echo esc_html__('WPForms', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post($this->link('https://upmlm.com/help/knowledgebase/wordpress-wp-form/', __('Instructions', 'upmlmconnect'))) . '</div>';
        echo '</div>';

        // Gravity Forms (link only)
        echo '<div style="padding:10px 0;border-bottom:1px solid #f0f0f1;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][gravityforms]" value="1" ' . checked(1, $providers['gravityforms'] ?? 0, false) . ' /> ';
        echo esc_html__('Gravity Forms', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post($this->link('https://upmlm.com/help/knowledgebase/wordpress-gravity-form/', __('Instructions', 'upmlmconnect'))) . '</div>';
        echo '</div>';

        // Forminator (link only)
        echo '<div style="padding:10px 0;border-bottom:1px solid #f0f0f1;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][forminator]" value="1" ' . checked(1, $providers['forminator'] ?? 0, false) . ' /> ';
        echo esc_html__('Forminator', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post($this->link('https://upmlm.com/help/knowledgebase/wordpress-forminator-form/', __('Instructions', 'upmlmconnect'))) . '</div>';
        echo '</div>';

        // Ninja Forms (link only)
        echo '<div style="padding:10px 0;">';
        echo '<label style="display:block;font-weight:600;">';
        echo '<input type="checkbox" name="' . esc_attr(self::OPTION_KEY) . '[providers][ninjaforms]" value="1" ' . checked(1, $providers['ninjaforms'] ?? 0, false) . ' /> ';
        echo esc_html__('Ninja Forms', 'upmlmconnect');
        echo '</label>';
        echo '<div class="description" style="margin-top:6px;">' . wp_kses_post($this->link('https://upmlm.com/help/knowledgebase/wordpress-ninja-form/', __('Instructions', 'upmlmconnect'))) . '</div>';
        echo '</div>';

        echo '</div>'; // provider box

        submit_button(__('Save settings', 'upmlmconnect'));

        $this->card_close();
        echo '</form>';

        // Test card
        $this->card_open(__('Connection test', 'upmlmconnect'));
        echo '<form method="post">';
        wp_nonce_field('upmlm_test_lead');
        echo '<input type="hidden" name="upmlm_test" value="1" />';
        submit_button(__('Send test lead', 'upmlmconnect'), 'secondary');
        echo '</form>';
        $this->card_close();

        // Log card (no full path shown)
        $this->card_open(__('Error log', 'upmlmconnect'));
        echo '<p class="description">' . esc_html__('Only API send failures are recorded.', 'upmlmconnect') . '</p>';
        echo '<textarea readonly style="width:100%;min-height:220px;font-family:ui-monospace,Menlo,Consolas,monospace;white-space:pre;">' . esc_textarea($this->tail_log(120)) . '</textarea>';
        echo '<form method="post" style="margin-top:10px;">';
        wp_nonce_field('upmlm_log_action');
        echo '<input type="hidden" name="upmlm_log_action" value="clear" />';
        submit_button(__('Clear log', 'upmlmconnect'), 'secondary');
        echo '</form>';
        $this->card_close();

        // Footer help
        echo '<div style="max-width:920px;margin-top:16px;">';
        echo '<strong>' . esc_html__('Need help?', 'upmlmconnect') . '</strong><br>';
        echo wp_kses_post(
            sprintf(
                __('Contact %s or visit %s', 'upmlmconnect'),
                '<a href="mailto:client@upmlm.com">client@upmlm.com</a>',
                $this->link('https://upmlm.com', 'https://upmlm.com')
            )
        );
        echo '</div>';

        echo '</div>'; // wrap
    }

    /* =====================
     * HOOKS
     * ===================== */

    public function hook_forms()
    {
        $settings = get_option(self::OPTION_KEY);
        if (empty($settings['enabled']))
            return;

        $providers = $settings['providers'] ?? [];

        if (!empty($providers['cf7']) && class_exists('WPCF7_Submission')) {
            add_action('wpcf7_mail_sent', [$this, 'handle_cf7']);
        }

        if (!empty($providers['elementor']) && class_exists('ElementorPro\\Plugin')) {
            add_action('elementor_pro/forms/new_record', [$this, 'handle_elementor'], 10, 2);
        }

        if (!empty($providers['wpforms'])) {
            add_action('wpforms_process_complete', [$this, 'handle_wpforms'], 10, 4);
        }

        if (!empty($providers['gravityforms'])) {
            add_action('gform_after_submission', [$this, 'handle_gravityforms'], 10, 2);
        }

        if (!empty($providers['forminator'])) {
            add_action('forminator_custom_form_submit_before_set_fields', [$this, 'handle_forminator'], 10, 1);
        }

        if (!empty($providers['ninjaforms'])) {
            add_action('ninja_forms_after_submission', [$this, 'handle_ninjaforms']);
        }
    }

    /* =====================
     * NORMALIZATION
     * ===================== */

    public function normalize_phone($phone)
    {
        $phone = trim((string) $phone);
        if ($phone === '')
            return null;
        $digits = preg_replace('/[^0-9]/', '', $phone);
        if ($digits === '')
            return null;
        return '+' . $digits;
    }

    private function normalize_lead_payload($name, $email, $phone = null)
    {
        $payload = [
            'name' => trim((string) $name),
            'email' => trim((string) $email),
        ];

        $phone = $this->normalize_phone($phone);
        if (!empty($phone)) {
            $payload['whatsapp_phone_number'] = $phone;
        }

        return $payload;
    }

    /* =====================
     * FIELD EXTRACTION (best-effort)
     * ===================== */

    private function pick_first_nonempty($candidates)
    {
        foreach ($candidates as $v) {
            if (is_string($v) && trim($v) !== '')
                return $v;
        }
        return '';
    }

    private function extract_from_kv($kv)
    {
        $name_candidates = [];
        $email_candidates = [];
        $phone_candidates = [];

        foreach ($kv as $k => $v) {
            $key = strtolower((string) $k);
            $val = is_array($v) ? implode(' ', array_map('strval', $v)) : (string) $v;

            if (strpos($key, 'email') !== false)
                $email_candidates[] = $val;
            if (strpos($key, 'phone') !== false || strpos($key, 'tel') !== false || strpos($key, 'mobile') !== false || strpos($key, 'whatsapp') !== false)
                $phone_candidates[] = $val;
            if (strpos($key, 'name') !== false || strpos($key, 'nome') !== false || strpos($key, 'first') !== false || strpos($key, 'last') !== false)
                $name_candidates[] = $val;
        }

        return [
            'name' => $this->pick_first_nonempty($name_candidates),
            'email' => $this->pick_first_nonempty($email_candidates),
            'phone' => $this->pick_first_nonempty($phone_candidates),
        ];
    }

    private function rgar_local($array, $key, $default = '')
    {
        if (!is_array($array))
            return $default;
        return array_key_exists($key, $array) ? $array[$key] : $default;
    }

    /* =====================
     * PROVIDER HANDLERS
     * ===================== */

    public function handle_cf7($contact_form)
    {
        $submission = WPCF7_Submission::get_instance();
        if (!$submission)
            return;

        $data = $submission->get_posted_data();

        $name = $data['your-name'] ?? ($data['name'] ?? '');
        $email = $data['your-email'] ?? ($data['email'] ?? '');
        $phone = $data['your-phone'] ?? ($data['phone'] ?? '');

        $payload = $this->normalize_lead_payload($name, $email, $phone);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = 'Contact Form 7';
        if (is_object($contact_form) && method_exists($contact_form, 'title')) {
            $t = (string) $contact_form->title();
            if (trim($t) !== '')
                $form_label = $t;
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'cf7');
    }

    public function handle_elementor($record, $handler)
    {
        $fields = [];
        foreach ((array) $record->get('fields') as $field) {
            if (!isset($field['id']))
                continue;
            $fields[$field['id']] = $field['value'] ?? '';
        }

        $name = $fields['name'] ?? '';
        $email = $fields['email'] ?? '';
        $phone = $fields['phone'] ?? '';

        if ($name === '' || $email === '') {
            $extracted = $this->extract_from_kv($fields);
            $name = $name ?: $extracted['name'];
            $email = $email ?: $extracted['email'];
            $phone = $phone ?: $extracted['phone'];
        }

        $payload = $this->normalize_lead_payload($name, $email, $phone);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = 'Elementor Form';
        if (is_object($record) && method_exists($record, 'get_form_settings')) {
            $n = (string) $record->get_form_settings('form_name');
            $id = $record->get_form_settings('id');
            if (trim($n) !== '') {
                $form_label = $n;
            } elseif (!empty($id)) {
                $form_label = 'Elementor Form #' . $id;
            }
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'elementor');
    }

    public function handle_wpforms($fields, $entry, $form_data, $entry_id)
    {
        $kv = [];
        foreach ((array) $fields as $field) {
            $id = $field['name'] ?? ($field['id'] ?? '');
            $label = $field['label'] ?? '';
            $type = $field['type'] ?? '';
            $value = $field['value'] ?? '';
            $key = $label ?: $id;

            if ($type === 'email')
                $kv['email'] = $value;
            elseif ($type === 'phone')
                $kv['phone'] = $value;
            elseif ($type === 'name')
                $kv['name'] = $value;
            else
                $kv[$key] = $value;
        }

        $extracted = $this->extract_from_kv($kv);

        $name = $kv['name'] ?? $extracted['name'];
        $email = $kv['email'] ?? $extracted['email'];
        $phone = $kv['phone'] ?? $extracted['phone'];

        $payload = $this->normalize_lead_payload($name, $email, $phone);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = $form_data['settings']['form_title'] ?? '';
        if (trim((string) $form_label) === '') {
            $fid = $form_data['id'] ?? '';
            $form_label = $fid ? ('WPForms #' . $fid) : 'WPForms';
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'wpforms');
    }

    public function handle_gravityforms($entry, $form)
    {
        $kv = [];
        foreach ((array) ($form['fields'] ?? []) as $field) {
            if (!is_object($field))
                continue;
            $id = (string) $field->id;
            $label = (string) $field->label;
            $type = (string) $field->type;

            $value = function_exists('rgar') ? rgar($entry, $id) : $this->rgar_local($entry, $id);
            if (is_array($value))
                $value = implode(' ', array_map('strval', $value));

            $key = $label ?: $id;

            if ($type === 'email')
                $kv['email'] = $value;
            elseif ($type === 'phone')
                $kv['phone'] = $value;
            elseif ($type === 'name')
                $kv['name'] = $value;
            else
                $kv[$key] = $value;
        }

        $extracted = $this->extract_from_kv($kv);

        $name = $kv['name'] ?? $extracted['name'];
        $email = $kv['email'] ?? $extracted['email'];
        $phone = $kv['phone'] ?? $extracted['phone'];

        $payload = $this->normalize_lead_payload($name, $email, $phone);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = $form['title'] ?? '';
        if (trim((string) $form_label) === '') {
            $fid = $form['id'] ?? '';
            $form_label = $fid ? ('Gravity Form #' . $fid) : 'Gravity Forms';
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'gravityforms');
    }

    public function handle_forminator($form_id)
    {
        $posted = wp_unslash($_POST);
        if (!is_array($posted))
            return;

        $extracted = $this->extract_from_kv($posted);
        $payload = $this->normalize_lead_payload($extracted['name'], $extracted['email'], $extracted['phone']);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = 'Forminator Form';
        if (!empty($form_id)) {
            $form_label = 'Forminator Form #' . $form_id;
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'forminator');
    }

    public function handle_ninjaforms($form_data)
    {
        $kv = [];
        if (!empty($form_data['fields']) && is_array($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                $label = $field['label'] ?? '';
                $key = $field['key'] ?? ($field['id'] ?? '');
                $type = $field['type'] ?? '';
                $value = $field['value'] ?? '';

                if ($type === 'email')
                    $kv['email'] = $value;
                elseif ($type === 'phone')
                    $kv['phone'] = $value;
                elseif (stripos($label . ' ' . $key, 'name') !== false)
                    $kv['name'] = $value;
                else
                    $kv[$label ?: $key] = $value;
            }
        }

        $extracted = $this->extract_from_kv($kv);

        $name = $kv['name'] ?? $extracted['name'];
        $email = $kv['email'] ?? $extracted['email'];
        $phone = $kv['phone'] ?? $extracted['phone'];

        $payload = $this->normalize_lead_payload($name, $email, $phone);
        if (empty($payload['name']) || empty($payload['email']))
            return;
        $form_label = 'Ninja Form';
        if (is_array($form_data)) {
            $fid = $form_data['form_id'] ?? ($form_data['id'] ?? '');
            if (!empty($fid))
                $form_label = 'Ninja Form #' . $fid;
        }
        $payload['about'] = $this->build_about($form_label);


        $this->send_to_upmlm($payload, false, 'ninjaforms');
    }

    /* =====================
     * API
     * ===================== */

    private function send_to_upmlm($payload, $show_notice = false, $source = '')
    {
        $settings = get_option(self::OPTION_KEY);

        $org_id = intval($settings['org_id'] ?? 0);
        $api_key = trim((string) ($settings['api_key'] ?? ''));

        if ($org_id <= 0 || $api_key === '') {
            $this->log_error($source, 'config', 'Missing Organization ID or API Key', $payload);
            update_option(self::STATUS_KEY, ['status' => 'error', 'time' => time()]);
            return false;
        }

        // Deduplication (60s) by email
        $dup_key = null;
        $email = strtolower(trim((string) ($payload['email'] ?? '')));
        if ($email !== '') {
            $dup_key = 'upmlm_sent_' . md5($email);
            if (get_transient($dup_key)) {
                return true; // silently ignore duplicate
            }
        }

        $url = 'https://app.upmlm.com/api/organizations/' . $org_id . '/leads';

        $args = [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($payload),
            'timeout' => 15,
        ];

        $response = wp_remote_post($url, $args);

        // Retry once on transient network error
        if (is_wp_error($response)) {
            usleep(300000); // 300ms
            $response = wp_remote_post($url, $args);
        }

        if (is_wp_error($response)) {
            $this->log_error($source, 'wp_error', $response->get_error_message(), $payload);
            update_option(self::STATUS_KEY, ['status' => 'error', 'time' => time()]);
            return false;
        }

        $code = (int) wp_remote_retrieve_response_code($response);
        if ($code !== 201) {
            $body = (string) wp_remote_retrieve_body($response);
            $this->log_error($source, 'http_' . $code, $body, $payload);
            update_option(self::STATUS_KEY, ['status' => 'error', 'time' => time()]);
            return false;
        }

        if ($dup_key) {
            set_transient($dup_key, true, 60);
        }

        update_option(self::STATUS_KEY, ['status' => 'success', 'time' => time()]);
        return true;
    }

    /* =====================
     * LOGGING
     * ===================== */

    private function get_log_path()
    {
        $uploads = wp_upload_dir();
        return trailingslashit($uploads['basedir']) . self::LOG_FILENAME;
    }

    private function ensure_log_dir_exists()
    {
        $uploads = wp_upload_dir();
        if (!empty($uploads['basedir']) && !file_exists($uploads['basedir'])) {
            wp_mkdir_p($uploads['basedir']);
        }
    }

    private function log_error($source, $type, $message, $payload = [])
    {
        $this->ensure_log_dir_exists();

        $line = sprintf(
            "[%s] source=%s type=%s message=%s payload=%s\n",
            gmdate('c'),
            $source ?: 'unknown',
            $type,
            preg_replace('/\s+/', ' ', trim((string) $message)),
            wp_json_encode($payload)
        );

        $path = $this->get_log_path();
        if (file_exists($path) && @filesize($path) > 2 * 1024 * 1024) {
            @rename($path, $path . '.1');
        }

        @file_put_contents($path, $line, FILE_APPEND | LOCK_EX);
    }

    private function tail_log($lines = 100)
    {
        $path = $this->get_log_path();
        if (!file_exists($path))
            return '';

        $content = @file($path, FILE_IGNORE_NEW_LINES);
        if (!is_array($content))
            return '';

        $slice = array_slice($content, -abs((int) $lines));
        return implode("\n", $slice);
    }

    private function clear_log()
    {
        $path = $this->get_log_path();
        if (file_exists($path)) {
            @file_put_contents($path, '');
        }
    }
}

new UpMlmConnect();
